/*
  C64 packer.

  block header's bits 1-0 indicate block tpye
    x 0 unpacked run
    0 1 RLE, bits 7-2 are length, repeat next char length times.
    1 1 equal strings, bits 7-2 are length, next word is backindex.

  Wotan mit uns.
*/


#include <fcntl.h>
#include <sys/stat.h>
#if defined(_WIN32) || defined(__DOS__) || defined(__MSDOS__)
#include <io.h>
#include <share.h>
#else
#include <unistd.h>
#endif
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>


// Defining this will result in some ugly statistics displayed.
//#define MAKE_LOG


#define MAX_BUFLEN      0xfff0L
#define MAX_UNPACKEDLEN 127
#define MAX_RLELEN      63
#define MAX_BACKINDEX   65535L
#define MAX_STRINGLEN   63

#define MAX_RLEMODELEN  255
#define RLE_ESCAPEBYTE  255

// Update progress indicator every PROGRESS_INDICATOR_RESOLUTION bytes.
#define PROGRESS_INDICATOR_RESOLUTION   1024

#ifndef FALSE
#define FALSE               0
#endif

#ifndef TRUE
#define TRUE                1
#endif


typedef int BOOL;
typedef unsigned char BYTE;             // exactly 8 bit unsigned
typedef unsigned short WORD;            // at least 16 bit unsigned


typedef struct {
  WORD next;                            // index of next array element.
  WORD index;                           // index in inbuf[];
} BYTEINDEX;


class C64Pack {
public:
  C64Pack(void);
  ~C64Pack(void);

  int load(const char *filename, BOOL skipheader=0);
  int pack(BOOL RLEMode=0);
  int depack(BOOL RLEMode=0);
  int save(const char *filename);

private:
  BYTE *inbuf;
  BYTE *outbuf;
  long inbufpos;
  long inbuflen;
  long outbuflen;

  long stringcounter[MAX_STRINGLEN+1];
  long lastindexed;
  WORD byteindices[256];
  BYTEINDEX *byteindexpool;
  WORD byteindexpoolpos;

  void updateByteIndices(long index);

  long getMaxRLELength(long start, long maxLength);
  long getMaxStringLength(long start, long& backindex);

  void writeByte(BYTE b);
  void writeUnpackedBlock(long len);
  void writeRLEBlock(long len);
  void writeStringBlock(long stringlen, long backindex);

  int packNormal(void);
  int packRLE(void);
  int depackNormal(void);
  int depackRLE(void);

// Open a file for reading with proper sharing.
  static int openForRead(const char *filename);
// Open a file for writing with proper sharing.
  static int openForWrite(const char *filename);

  static void printHelp(FILE *f);
};


#ifndef getopt
// POSIX getopt function from DJGPP
// Watcom C++ and Visual C++ does not have getopt(). Lame.

int opterr = 1, optind = 1, optopt = 0;
char *optarg = 0;

#define BADCH   (int)'?'
#define EMSG    ""

int getopt(int nargc, char *const nargv[], const char *ostr)
{
  static char *place = EMSG;            // option letter processing
  char *oli;                            // option letter list index
  char *p;

  if (!*place) {
    if (optind >= nargc || *(place = nargv[optind]) != '-') {
      place = EMSG;
      return(EOF);
    }
    if (place[1] && *++place == '-') {
      ++optind;
      place = EMSG;
      return(EOF);
    }
  }

  if ((optopt = (int)*place++) == (int)':' || !(oli = strchr(ostr, optopt))) {
    // if the user didn't specify '-' as an option, assume it means EOF.
    if (optopt == (int)'-') return EOF;
    if (!*place) ++optind;
    if (opterr) {
      if (!(p = strrchr(*nargv, '/'))) p = *nargv;
      else ++p;
      fprintf(stderr, "%s: illegal option -- %c\n", p, optopt);
    }
    return BADCH;
  }
  if (*++oli != ':') {
    // don't need argument
    optarg = NULL;
    if (!*place) ++optind;
  }
  else {
    // need an argument
    if (*place) {}                      // no white space
    else if (nargc <= ++optind) {
      // no arg
      place = EMSG;
      if (!(p = strrchr(*nargv, '/'))) p = *nargv;
      else ++p;
      if (opterr) fprintf(stderr, "%s: option requires an argument -- %c\n", p, optopt);
      return BADCH;
    }
    else optarg = nargv[optind];        // white space
    place = EMSG;
    ++optind;
  }
  return optopt;                        // dump back option letter
}
#endif


// Open a file for reading with proper sharing.
int C64Pack::openForRead(const char *filename)
{
#if defined(_WIN32) || defined(__DOS__) || defined(__MSDOS__)
  return sopen(filename, O_RDONLY | O_BINARY, SH_DENYNO, S_IREAD);
#else
  return open(filename, O_RDONLY, S_IREAD);
#endif
}


// Open a file for writing with proper sharing.
int C64Pack::openForWrite(const char *filename)
{
#if defined(_WIN32) || defined(__DOS__) || defined(__MSDOS__)
  return sopen(filename, O_WRONLY | O_BINARY | O_CREAT, SH_DENYWR, S_IWRITE);
#else
  return open(filename, O_CREAT | O_TRUNC | O_WRONLY, S_IWUSR | S_IRUSR | S_IRGRP | S_IROTH);
#endif
}


C64Pack::C64Pack(void)
{
  inbuf=new BYTE[MAX_BUFLEN];
  outbuf=new BYTE[MAX_BUFLEN];
}


C64Pack::~C64Pack(void)
{
  delete inbuf;
  delete outbuf;
}


void C64Pack::updateByteIndices(long index)
{
  while (lastindexed<index) {
    lastindexed++;
    BYTE thisbyte=inbuf[lastindexed];
// Register this byte into the byte pool.
    byteindexpool[byteindexpoolpos].next=byteindices[thisbyte];
    byteindexpool[byteindexpoolpos].index=(WORD)lastindexed;
    byteindices[thisbyte]=byteindexpoolpos;
    byteindexpoolpos++;
  }
}


long C64Pack::getMaxRLELength(long start, long maxLength)
{
  long end=inbuflen-start;
  if (end>maxLength) end=maxLength;
  BYTE c=inbuf[start];
  long len;
  for (len=1; len<end; len++) {
    if (c!=inbuf[start+len]) break;
  }
  return len;
}


long C64Pack::getMaxStringLength(long start, long& backindex)
{
  long bestlen=0;
  long bestindex=0;
  WORD byteindex=byteindices[inbuf[start]];
  while (byteindex!=0xffff) {
    long oldindex=byteindexpool[byteindex].index;
    BYTE *curr=inbuf+start;
    BYTE *old=inbuf+oldindex;
    if (curr-old<=MAX_BACKINDEX) {
      long maxlen=start-oldindex;
      if (maxlen>MAX_STRINGLEN) maxlen=MAX_STRINGLEN;
      if (maxlen>inbuflen-start) maxlen=inbuflen-start;   // Don't go past EOF.
      long l;
      for (l=0; l<maxlen; l++, curr++, old++) {
        if ((*curr)!=(*old)) break;
      }
      if (bestlen<l) {
        bestlen=l;
        bestindex=oldindex;
      }
    }
    byteindex=byteindexpool[byteindex].next;
  }
  backindex=start-bestindex;
  return bestlen;
}


void C64Pack::writeByte(BYTE b)
{
  outbuf[outbuflen]=b;
  outbuflen++;
}


void C64Pack::writeUnpackedBlock(long len)
{
  writeByte((BYTE)(len<<1));
  for (long i=0; i<len; i++) {
    writeByte(inbuf[inbufpos]);
    inbufpos++;
  }
}


void C64Pack::writeRLEBlock(long rlelen)
{
  writeByte(((BYTE)(rlelen<<2)+1));
  writeByte(inbuf[inbufpos]);
  inbufpos+=rlelen;
}


void C64Pack::writeStringBlock(long stringlen, long backindex)
{
  backindex=-backindex;
  writeByte((stringlen<<2)+3);
  writeByte(backindex&0xff);
  writeByte((backindex>>8)&0xff);
  inbufpos+=stringlen;
}


int C64Pack::packNormal(void)
{
  inbufpos=0;
  outbuflen=0;

// Init byte index lists.
  for (int b=0; b<256; b++) byteindices[b]=0xffff;
  lastindexed=-1;
  byteindexpool=new BYTEINDEX[inbuflen];
  byteindexpoolpos=0;

  for (long s=0; s<MAX_STRINGLEN+1; s++) {
    stringcounter[s]=0;
  }

// Latest inbuf position where progress bar was updated.
  long inbufpos_latest = 0;
  for (;;) {
    if ((inbuflen>0) && (inbufpos>=inbufpos_latest+PROGRESS_INDICATOR_RESOLUTION)) {
      fprintf(stderr, "%6.2f%%\r", (100.0f*inbufpos)/inbuflen);
      inbufpos_latest=inbufpos;
    }
    long end=inbuflen-inbufpos;
    if (end==0) break;
    if (end<0) {
      fprintf(stderr, "Bug in packer. Burn in hell.\n");
      exit(EXIT_FAILURE);
    }
    if (end>MAX_UNPACKEDLEN) end=MAX_UNPACKEDLEN;
// Init run and string lengths.
    long rlelen=0;
    long stringlen=0;
    long backindex;
    long s;
    for (s=0; s<end; s++) {
      updateByteIndices(inbufpos+s);
// Get maximum run length.
      rlelen=getMaxRLELength(inbufpos+s, MAX_RLELEN);
      if (rlelen==MAX_RLELEN) break;
// Get maximum string length.
      stringlen=getMaxStringLength(inbufpos+s, backindex);
      if (rlelen>=3) break;
      if (stringlen>=4) break;
    }
    if (s) writeUnpackedBlock(s);
    if (rlelen>=stringlen) {
      if (rlelen>=3) writeRLEBlock(rlelen);
    }
    else {
      if (stringlen>=4) writeStringBlock(stringlen, backindex);
    }
    stringcounter[stringlen]++;
#ifdef MAKE_LOG
    fprintf(stdout, "%6ld %4ld\n", backindex, stringlen);
#endif
  }
  writeUnpackedBlock(0);
  if (inbuflen>0) fprintf(stderr, "%6.2f%%\n", (100.0f*outbuflen)/inbuflen);

#ifdef MAKE_LOG
  for (s=0; s<MAX_STRINGLEN+1; s++) {
    printf("%03ld %04ld\n", s, stringcounter[s]);
  }
#endif

// Free byte index lists.
  delete byteindexpool;
  return 0;
}


int C64Pack::packRLE(void)
{
  inbufpos=0;
  outbuflen=0;

// Write header "RL".
  writeByte(0x52);
  writeByte(0x4c);

// Latest inbuf position where progress bar was updated.
  long inbufpos_latest = 0;
  for (;;) {
    if ((inbuflen>0) && (inbufpos>=inbufpos_latest+PROGRESS_INDICATOR_RESOLUTION)) {
      fprintf(stderr, "%6.2f%%\r", (100.0f*inbufpos)/inbuflen);
      inbufpos_latest=inbufpos;
    }
    long end=inbuflen-inbufpos;
    if (end==0) break;
    if (end<0) {
      fprintf(stderr, "Bug in packer. Burn in hell.\n");
      exit(EXIT_FAILURE);
    }
// Get maximum run length.
    long rlelen=getMaxRLELength(inbufpos, MAX_RLEMODELEN);
// If this is the escape byte, always write RLE block.
    if (inbuf[inbufpos]==RLE_ESCAPEBYTE) {
      writeByte(RLE_ESCAPEBYTE);
      writeByte((BYTE)rlelen);
      writeByte(inbuf[inbufpos]);
      inbufpos+=rlelen;
    }
// Else write RLE only if run is more than 3 characters long.
    else {
      if (rlelen>3) {
        writeByte(RLE_ESCAPEBYTE);
        writeByte((BYTE)rlelen);
        writeByte(inbuf[inbufpos]);
        inbufpos+=rlelen;
      }
      else {
        writeByte(inbuf[inbufpos]);
        inbufpos++;
      }
    }
  }
// Signal end of file.
  writeByte(RLE_ESCAPEBYTE);
  writeByte(0);
  writeByte(0);

  if (inbuflen>0) fprintf(stderr, "%6.2f%%\n", (100.0f*outbuflen)/inbuflen);
  return 0;
}


int C64Pack::pack(BOOL RLEMode)
{
  if (RLEMode) return packRLE();
  else return packNormal();
}


int C64Pack::depackNormal(void)
{
  inbufpos=0;
  outbuflen=0;

  for (;;) {
    BYTE c=inbuf[inbufpos];
    if (c==0) break;
    inbufpos++;
// Uncompressed block.
    if ((c&1)==0) {
      long l=c>>1;
      for (long i=0; i<l; i++) {
        outbuf[outbuflen]=inbuf[inbufpos];
        inbufpos++;
        outbuflen++;
      }
    }
    else {
// RLE block.
      if ((c&2)==0) {
        long l=c>>2;
        c=inbuf[inbufpos];
        inbufpos++;
        for (long i=0; i<l; i++) {
          outbuf[outbuflen]=c;
          outbuflen++;
        }
      }
// String block.
      else {
        long l=c>>2;
        long bi=inbuf[inbufpos];
        inbufpos++;
        bi|=inbuf[inbufpos]<<8;
        inbufpos++;
        for (long i=0; i<l; i++) {
          outbuf[outbuflen]=outbuf[outbuflen+bi-0x10000];
          outbuflen++;
        }
      }
    }
  }

  return 0;
}


int C64Pack::depackRLE(void)
{
  if ((inbuf[0] == 0x52) && (inbuf[1] == 0x4c)) {
    inbufpos = 2;
    outbuflen = 0;
    for (;;) {
      BYTE c = inbuf[inbufpos];
      inbufpos++;
// RLE block.
      if (c == RLE_ESCAPEBYTE) {
        long l = inbuf[inbufpos];
        inbufpos++;
        if (l == 0) break;
        c = inbuf[inbufpos];
        inbufpos++;
        for (long i = 0; i < l; i++) {
          outbuf[outbuflen] = c;
          outbuflen++;
        }
      }
// Single character.
      else {
        outbuf[outbuflen] = c;
        outbuflen++;
      }
    }
  }
  else {
    memmove(outbuf, inbuf, inbuflen);
    outbuflen = inbuflen;
  }
  return 0;
}


int C64Pack::depack(BOOL RLEMode)
{
  if (RLEMode) return depackRLE();
  else return depackNormal();
}


int C64Pack::load(const char *filename, BOOL skipheader)
{
  int fd=openForRead(filename);
  if (fd==-1) return -1;
  if (skipheader) lseek(fd, 2, SEEK_CUR);
  inbuflen=read(fd, inbuf, MAX_BUFLEN);
  if (inbuflen==-1) return -1;
  return close(fd);
}


int C64Pack::save(const char *filename)
{
  int fd=openForWrite(filename);
  if (fd==-1) return -1;
  write(fd, outbuf, outbuflen);
  return close(fd);
}


void printHelp(FILE *f)
{
  fprintf(f, "c64pack command [switches] inputfile outputfile\n");
  fprintf(f, "commands:  e    pack (encode)\n");
  fprintf(f, "           d    depack (decode)\n");
  fprintf(f, "switches: -s    skip load address (2 bytes) in input\n");
  fprintf(f, "          -r    RLE pack/depack\n");
}


int main(int argc, char *argv[])
{
  if (argc<4) printHelp(stderr);
  else {
// Parse command.
    int command=0;
    if (strcmp(argv[1], "e")==0) command='e';
    if (strcmp(argv[1], "d")==0) command='d';
    if (command==0) {
      fprintf(stderr, "Unknown command %s.\n", argv[1]);
      exit(1);
    }
// Parse options.
    int i;
    BOOL skipLoadAddress=FALSE;
    BOOL RLEMode=FALSE;
    for (i=2; i<argc; i++) {
      if (argv[i][0]!='-') break;
      if (strcmp(argv[i], "-s")==0) skipLoadAddress=TRUE;
      else if (strcmp(argv[i], "-r")==0) RLEMode=TRUE;
      else {
        fprintf(stderr, "Unknown option %s.\n", argv[i]);
        return EXIT_FAILURE;
      }
    }
// Check if there are 2 more options. (input and output filenames)
        if (argc != i+2) {
          printHelp(stderr);
      return EXIT_FAILURE;
        }
// And go.
    C64Pack p;
    if (p.load(argv[i], skipLoadAddress)==-1) {
      fprintf(stderr, "Unable to load input file.\n");
      return EXIT_FAILURE;
    }
    int result=0;
    switch (command) {
      case 'e': result=p.pack(RLEMode); break;
      case 'd': result=p.depack(RLEMode); break;
    }
    if (result) {
      fprintf(stderr, "Error while packing/depacking. Doom found you.\n");
      return EXIT_FAILURE;
    }
// Save output.
    i++;
    if (p.save(argv[i])==-1) {
      fprintf(stderr, "Unable to save output.\n");
      return EXIT_FAILURE;
    }
  }

  return EXIT_SUCCESS;
}
