/*
  TGA to C64 bitmap/font converter.

  argv[1]: name of 8 bit uncompressed TGA to read
  argv[2]: name of C64 bitmap/font image to write

  Wotan mit uns.
*/


#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#ifndef FALSE
#define FALSE               0
#endif

#ifndef TRUE
#define TRUE                1
#endif


typedef int BOOL;
typedef unsigned char BYTE;             // 8 bit unsigned
typedef unsigned short WORD;            // 16 bit unsigned
typedef unsigned long DWORD;            // 32 bit unsigned


#define TGA_HEADER_SIZE                 18


class TGA2Bitmap {
public:
  TGA2Bitmap(void);
  ~TGA2Bitmap(void);

  int read(const char *filename, BOOL flip=FALSE);
  int writeC64Bitmap(const char *filename);

private:
  typedef struct {
    BYTE b;
    BYTE g;
    BYTE r;
  } PALETTE;

  typedef struct {
    BYTE idfieldlen;
    BYTE colormaptype;
    BYTE imagetype;
    WORD firstcolor;
    WORD colorsnum;
    BYTE bitspercolor;
    WORD xorig;
    WORD yorig;
    WORD width;
    WORD height;
    BYTE bitsperpixel;
    BYTE desc;
  } TGA_HEADER;

  TGA_HEADER tga;

  int imagetype;
  int width;
  int height;
  int bpp;
  int cmaplen;

  PALETTE pal[256];
  BYTE *pixels;
  int c64Width;
  int c64Height;
  BYTE *c64Bitmap;

  int convertToC64Bitmap(void);

// File interface.
  FILE *f;

  int open(const char *filename);
  int close(void);
  long tell(void);
  int seek(long n);
  int advance(long n);

  int getByte(BYTE& t);
  int getWord(WORD& t);
  int getDWord(DWORD& t);
};


TGA2Bitmap::TGA2Bitmap(void)
{
  f=NULL;
  pixels=NULL;
  c64Bitmap=NULL;
}


TGA2Bitmap::~TGA2Bitmap(void)
{
  close();
  if (pixels) {
    delete pixels;
    pixels=NULL;
  }
  if (c64Bitmap) {
    delete c64Bitmap;
    c64Bitmap=NULL;
  }
}


int TGA2Bitmap::open(const char *filename)
{
  if (!f) {
    f=fopen(filename, "rb");
    if (f) return 0;
    return -1;
  }
  return -1;
}


int TGA2Bitmap::close(void)
{
  if (f) {
    int r=fclose(f);
    if (r) return -1;
    else {
      f=NULL;
      return 0;
    }
  }
  return -1;
}


// Seek to a position.
//
//  Input:     n:       offset from beginning of file
// Output: [ret]:       0 if fseek() was OK, nonzero otherwise
int TGA2Bitmap::seek(long n)
{
  return fseek(f, n, SEEK_SET);
}


// Advance by n bytes.
//
//  Input:     n:       number of bytes to advance
// Output: [ret]:       0 if fseek() was OK, nonzero otherwise
int TGA2Bitmap::advance(long n)
{
  return fseek(f, n, SEEK_CUR);
}


// Read an unsigned byte from the file.
// Exits program if read failed.
//
// Output: [ret]:       unsigned byte read from current position
int TGA2Bitmap::getByte(BYTE& t)
{
  return (fread(&t, sizeof(t), 1, f)==1) ? 0 : -1;
}


// Read an unsigned word (16 bits) from the file.
// Exits program if read failed.
//
// Output: [ret]:       unsigned word read from current position
int TGA2Bitmap::getWord(WORD& t)
{
  return (fread(&t, sizeof(t), 1, f)==1) ? 0 : -1;
}


// Read an unsigned double word (32 bits) from the file.
// Exits program if read failed.
//
// Output: [ret]:       unsigned double word read from current position
int TGA2Bitmap::getDWord(DWORD& t)
{
  return (fread(&t, sizeof(t), 1, f)==1) ? 0 : -1;
}


int TGA2Bitmap::read(const char *filename, BOOL flip)
{
  int bitmapsrc, palsrc;
  int i;
  int y;

  if (open(filename)) return -1;
// Read header.
  if (getByte(tga.idfieldlen)) return -1;
  if (getByte(tga.colormaptype)) return -1;
  if (getByte(tga.imagetype)) return -1;
  if (getWord(tga.firstcolor)) return -1;
  if (getWord(tga.colorsnum)) return -1;
  if (getByte(tga.bitspercolor)) return -1;
  if (getWord(tga.xorig)) return -1;
  if (getWord(tga.yorig)) return -1;
  if (getWord(tga.width)) return -1;
  if (getWord(tga.height)) return -1;
  if (getByte(tga.bitsperpixel)) return -1;
  if (getByte(tga.desc)) return -1;
// Extract relevant information from the header.
  imagetype=tga.imagetype;
  width=tga.width;
  height=tga.height;
  switch (imagetype) {
// Paletted file.
    case 1:
      bpp=8;
      cmaplen=tga.colorsnum;
      break;
// Greyscale file.
    case 3:
      bpp=8;
      cmaplen=256;
      break;
// Unknown file.
    default:
      return -1;
  }
  pixels=new BYTE[width*height];
  memset(pixels, 0, width*height*sizeof(BYTE));

  switch (imagetype) {
// Paletted file.
    case 1:
      palsrc=TGA_HEADER_SIZE+tga.idfieldlen;
      seek(palsrc);
      for (i=0; i<tga.colorsnum; i++) {
        WORD c;
        switch (tga.bitspercolor) {
          case 16:
            if (getWord(c)) return -1;
            pal[i].r=((c>>10)&0x1f)<<3;
            pal[i].g=((c>>5)&0x1f)<<3;
            pal[i].b=(c&0x1f)<<3;
            break;
          case 24:
            if (getByte(pal[i].b)) return -1;
            if (getByte(pal[i].g)) return -1;
            if (getByte(pal[i].r)) return -1;
            break;
          case 32:
            if (getByte(pal[i].b)) return -1;
            if (getByte(pal[i].g)) return -1;
            if (getByte(pal[i].r)) return -1;
            advance(1);
            break;
          default:              // This is impossible, just shut up compiler.
            if (getByte(pal[i].b)) return -1;
            if (getByte(pal[i].g)) return -1;
            if (getByte(pal[i].r)) return -1;
            advance(1);
        }
      }
      bitmapsrc=TGA_HEADER_SIZE+tga.idfieldlen+tga.colorsnum*tga.bitspercolor/8;
      seek(bitmapsrc);
      for (y=0; y<height; y++) {
// Is image flipped?
        int d;
        if (((tga.desc&0x20)==0x20)^flip) d=y;
        else d=height-1-y;
        for (int x=0; x<width; x++) {
          if (getByte(pixels[d*width+x])) return -1;
        }
      }
      break;
// Grayscale file.
    case 3:
      bitmapsrc=TGA_HEADER_SIZE+tga.idfieldlen+tga.colorsnum*tga.bitspercolor/8;
      for (y=0; y<height; y++) {
// Is image flipped?
        int d;
        if (((tga.desc&0x20)==0x20)^flip) d=y;
        else d=height-1-y;
        for (int x=0; x<width; x++) {
          if (getByte(pixels[d*width+x])) return -1;
        }
      }
      break;
  }

  return convertToC64Bitmap();
}


int TGA2Bitmap::convertToC64Bitmap(void)
{
  if (!pixels) return -1;
  c64Width=(width+7)&0xff8;
  c64Height=(height+7)&0xff8;
  c64Bitmap=new BYTE[c64Width/8*c64Height];
  memset(c64Bitmap, 0, c64Width/8*c64Height*sizeof(BYTE));

  for (int y=0; y<c64Height; y++) {             // Loop pixel positions Y.
    for (int xc=0; xc<c64Width; xc+=8) {        // Loop character positions X.
      int m=128;                      // Bit mask for current x pixel pos.
      int t=0;                        // Bits collected so far.
      for (int xp=0; xp<8; xp++) {    // Loop pixel positions X.
        int p=0;
// Read pixel p only if it is inside the original TGA.
        if ((y<height)&&((xc+xp)<width)) p=pixels[y*width+xc+xp];
        if (p) t|=m;
        m>>=1;
      }
      int yc=(y/8)*c64Width;
      int yp=y&7;
      c64Bitmap[yc+xc+yp]=t;
    }
  }

  return 0;
}


int TGA2Bitmap::writeC64Bitmap(const char *filename)
{
  if (!c64Bitmap) return -1;
  FILE *out=fopen(filename, "wb");
  if (!out) return -1;
  fwrite(c64Bitmap, c64Width/8*c64Height*sizeof(BYTE), 1, out);
  fclose(out);
  return 0;
}


int main(int argc, char *argv[])
{
  TGA2Bitmap *t;

  if (argc==3) {
    t=new TGA2Bitmap();
    if (t->read(argv[1])) {
      fprintf(stderr, "Error reading input file.");
      delete t;
      return EXIT_FAILURE;
    }
    if (t->writeC64Bitmap(argv[2])) {
      fprintf(stderr, "Error writing output file.");
      delete t;
      return EXIT_FAILURE;
    }
    delete t;
  }
  return EXIT_SUCCESS;
}
