/* makerom.c - Creates BBG ROM image from Novaterm files

   The ROM memory is used as follows:

   pages  used for
   -----  --------
     0    Initializer/ROM boot program
    1-6   Default buffer directory
    7-14  Title screen (twice)
  15-122  main program
 123-140  ML files, RAM driver
   141+   files
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define PRG  0
#define SEQ  1
#define SCR  2

#define FLAGS  0x84

#define INITPROG   "s-init.ml"
#define ROMBOOT    "romboot.ml"

#define TITLE      "title.c64"
#define NOVAML1    "nova9.6-1.ml"
#define NOVAML2    "nova9.6-2.ml"
#define NOVAPRG    "novaterm 9.6"
#define RAMDRIVER  "ram.default"

#define ROMSIZE    256
#define DIRSIZE    7
#define BOOTSIZE   768
#define VECTOR     0x2000

struct novafile {
  char *name;
  char *dname;
  char type;
};

struct bufent {
  char *name;
  int start;
  int end;
  short type;
};


struct novafile file[]=
{
  { "novaterm 9.6", "b", PRG },
  { "nova9.6-1.ml", "c", PRG },
  { "nova9.6-2.ml", "d", PRG },
  { "nova9.6-rtl.ml", "d1", PRG },
  { "term.ANSI", "e", PRG },
  { "term.VT102", "f", PRG },
  { "term.VT52", "g", PRG },
  { "term.standard", "h", PRG },
  { "term.ANSI-40col", "i", PRG },
  { "asc.standard", "j", PRG },
  { "asc.ANSI", "k", PRG },
  { "asc.VT", "ka", PRG },
  { "asc.ANSI-40col", "l", PRG },
  { "font80.novaterm", "m", PRG },
  { "font80.nova wide", "n", PRG },
  { "font80.ANSI", "o", PRG },
  { "font80.ANSI-wide", "p", PRG },
  { "font80.VT102+52", "r", PRG },
  { "font80.VT wide", "s", PRG },
  { "font81.ANSI", "f1", PRG },
  { "font81.VT102+52", "f2", PRG },
  { "font81.novaterm", "f3", PRG },
  { "font.novaterm", "u", PRG },
  { "font.ANSI-40col", "v", PRG },
  { "prt.Zmodem", "w", PRG },
  { "prt.Ymodem batch", "x", PRG },
  { "prt.Ymodem-g", "xa", PRG },
  { "prt.Xmodem-CRC", "y", PRG },
  { "prt.Xmodem-1k", "z", PRG },
  { "prt.Xmodem-1k-g", "ya", PRG },
  { "prt.Punter", "za", PRG },
  { "prt.Multi-Punter", "zb", PRG },
  { "prt.Kermit", "zc", PRG },
  { "prt.WXmodem recv", "zd", PRG },
  { "modem.1650", "ze", PRG },
  { "modem.1660", "zf", PRG },
  { "modem.Hayes 1200", "zh", PRG },
  { "modem.Hayes 2400", "zi", PRG },
  { "modem.Hayes 9600", "zj", PRG },
  { "modem.Hayes fast", "zl", PRG },
  { "ram.internal", "ra1", PRG },
  { "ram.VDC+internal", "rb", PRG },
  { "ram.VDC-alt+int", "rb1", PRG },
  { "ram.REU", "rc", PRG },
  { "ram.BBG-GEORam", "rd", PRG },
  { "ram.RAMDrive", "re", PRG },
  { "ram.RAMLink", "rf", PRG },
  { "80col.default", "pa", PRG },
  { "80col.fast-REU", "pb", PRG },
  { "80col.VDC (25)", "pc", PRG },
  { "80col.VDC (28)", "pd", PRG },
  { "serial.User port", "sa", PRG },
  { "serial.CommPort", "sb", PRG },
  { "serial.CP-DE00", "sb1", PRG },
  { "serial.CP-DE20", "sb2", PRG },
  { "serial.CP-DF20", "sb3", PRG },
  { "serial.CP-D700", "sb4", PRG },
  { "serial.SwiftLink", "sc", PRG },
  { "serial.SL-DF00", "sc1", PRG },
  { "serial.SL-DE20", "sc2", PRG },
  { "serial.SL-DF20", "sc3", PRG },
  { "serial.SL-D700", "sc4", PRG },
  { "serial.HART", "sd", PRG },
  { "serial.UP9600", "se", PRG },
  { "time.Manual", "ta", PRG },
  { "time.CMD-RTC", "tb", PRG },
  { "time.BBRTC-port1", "tc", PRG },
  { "time.BBRTC-port2", "td", PRG },
  { "terminal help", "zn", SEQ },
  { "nova.Text editor", "zo1", PRG },
  { "nova.BBS mode", "zr1", PRG },
  { "nova.ASCII edit", "zt1", PRG },
  { "nova.Font editor", "zu1", PRG },
  { "nova.Configure", "zy1", PRG },
  { "", "", PRG }
};

struct bufent* entry;

void putint(int val, char intlen, FILE* fo)
{
  unsigned char *sv = (unsigned char*)&val;
  register char i;

  for(i=0; i<intlen; i++)
    fputc(sv[i], fo);
}

void putbufent(struct bufent* ent, FILE* fo)
{
  register short i;
  char c;

  putint(ent->start, 3, fo);
  putint(ent->end, 3, fo);
  fputc(ent->type, fo);

  for(i=0; i<strlen(ent->name); i++) {
    c = ent->name[i];
    if(c >= 65 && c <= 90)
      c += 128;
    else if(c >= 97 && c <= 122)
      c -= 32;

    fputc(c, fo);
  }

  for(i=strlen(ent->name); i<16; i++)
    fputc(160, fo);
}

void main(int argc, char **argv)
{
  register int i;
  int maxfil, size, bufpos, s;
  FILE *fi, *fo;
  char buf[30000];

  int start_novaml1, end_novaml1;
  int start_novaml2, end_novaml2;
  int start_novaprg, end_novaprg;
  int start_ramdriver, end_ramdriver;
  int end_rom;

  if(argc < 2)
    exit(1);

  fo = fopen(argv[1], "w+");
  if(!fo)
    exit(1);

  memset(buf, 0, 30000);

/* Write tags and copyright message */

  strcpy(buf, "NOVATERM 9.6  COPYRIGHT 1996 BY NICK ROSSI  HARDWARE BY PERFORMANCE PERIPHERALS, INC.  FIGHT GOVERNMENT INTERFERENCE ON THE INTERNET! (AND IN EVERYTHING ELSE, FOR THAT MATTER)");

  fwrite(buf, sizeof(char), 256-2, fo);

  for(i=0; strlen(file[i].name); i++);
  maxfil = i;

  fputc(maxfil, fo);
  fputc(ROMSIZE>>6, fo);

/* Write blank space for directory entries */

  bufpos = ftell(fo);

  memset(buf, 0, DIRSIZE<<8);
  fwrite(buf, sizeof(char), DIRSIZE<<8, fo);

/* Write initializer into first page (256 bytes max) */

  fi = fopen(INITPROG, "r");
  if(!fi) {
    fprintf(stderr, "Initializer program not found\n");
    exit(1);
  }

  size = fread(buf, sizeof(char), 256, fi) - 2;
  fwrite(&buf[2], sizeof(char), size, fo);

  fclose(fi);

/* Write ROM boot into same page (256 bytes max) */

  fi = fopen(ROMBOOT, "r");
  if(!fi) {
    fprintf(stderr, "ROM boot program not found\n");
    exit(1);
  }

  memset(buf, 0, BOOTSIZE);
  fread(buf, sizeof(char), BOOTSIZE-size, fi);
  fwrite(&buf[2], sizeof(char), BOOTSIZE-size, fo);

  fclose(fi);

/* Pad out to $A000 */

  memset(buf, 0, 30000);

  size = ftell(fo);
  fwrite(buf, sizeof(char), VECTOR-size, fo);

/* Start address: $8806 */

  fputc(0x00, fo);
  fputc(0x88, fo);

  fwrite(buf, sizeof(char), 256-2, fo);

  entry = (struct bufent*)malloc(maxfil * sizeof(struct bufent));

  for(i=0; i<maxfil; i++)
    entry[i].name = strdup(file[i].name);

/* Write title screen twice */

  fi = fopen(TITLE, "r");
  if(!fi) {
    fprintf(stderr, "%s not found\n", TITLE);
    exit(1);
  }

  size = fread(buf, sizeof(char), 8192, fi);
  putint(size-2, 2, fo);
  fwrite(buf, sizeof(char), size, fo);

  fclose(fi);

/* Write main programs */

  memset(buf, 0, 30000);

  fi = fopen(NOVAML1, "r");
  if(!fi) {
    fprintf(stderr, "%s not found\n", NOVAML1);
    exit(1);
  }

  size = fread(buf, sizeof(char), 30000, fi);

  putint(size-2, 2, fo);
  start_novaml1 = ftell(fo);
  fwrite(buf, sizeof(char), size, fo);
  end_novaml1 = ftell(fo);

  fclose(fi);

  fi = fopen(NOVAML2, "r");
  if(!fi) {
    fprintf(stderr, "%s not found\n", NOVAML2);
    exit(1);
  }

  size = fread(buf, sizeof(char), 30000, fi);

  putint(size-2, 2, fo);
  start_novaml2 = ftell(fo);
  fwrite(buf, sizeof(char), size, fo);
  end_novaml2 = ftell(fo);

  fclose(fi);

  fi = fopen(RAMDRIVER, "r");
  if(!fi) {
    fprintf(stderr, "%s not found\n", RAMDRIVER);
    exit(1);
  }

  size = fread(buf, sizeof(char), 30000, fi);
  
  putint(size-2, 2, fo);
  start_ramdriver = ftell(fo);
  fwrite(buf, sizeof(char), size, fo);
  end_ramdriver = ftell(fo);

  fclose(fi);

  fprintf(stderr, "Start of program: %d,%d (%d)\n", end_ramdriver / 256, end_ramdriver % 256, end_ramdriver);

  fi = fopen(NOVAPRG, "r");
  if(!fi) {
    fprintf(stderr, "%s not found\n", NOVAPRG);
    exit(1);
  }

  size = fread(buf, sizeof(char), 30000, fi);

  putint(size-2, 2, fo);
  start_novaprg = ftell(fo);
  fwrite(buf, sizeof(char), size, fo);
  end_novaprg = start_novaprg + size;

  fclose(fi);

/* Pad file to next page */

  memset(buf, 0, 30000);
  fwrite(buf, sizeof(char), 38000 - end_novaprg, fo);

  fprintf(stderr, "End of main stuff: %d,%d (%d)\n", end_novaprg / 256, end_novaprg % 256, end_novaprg);

  for(i=0; i<maxfil; i++) {
    if(!strcmp(entry[i].name, NOVAML1)) {
      entry[i].start = start_novaml1;
      entry[i].end = end_novaml1;
    }
    else if(!strcmp(entry[i].name, NOVAML2)) {
      entry[i].start = start_novaml2;
      entry[i].end = end_novaml2;
    }
    else if(!strcmp(entry[i].name, NOVAPRG)) {
      entry[i].start = start_novaprg;
      entry[i].end = end_novaprg;
    }
    else if(!strcmp(entry[i].name, RAMDRIVER)) {
      entry[i].start = start_ramdriver;
      entry[i].end = end_ramdriver;
    }
    else {
      if(!strlen(file[i].dname)) {
	entry[i].start = entry[i-1].start;
	entry[i].end = entry[i-1].end;
	fprintf(stderr, "Making duplicate file: %s = %s\n", entry[i].name, entry[i-1].name);
      }
      else {
	fi = fopen(file[i].dname, "r");
	if(!fi) {
	  fprintf(stderr, "File not found: %s (%s)\n", file[i].name, file[i].dname);
	  exit(1);
	}

	entry[i].start = ftell(fo);

	size = fread(buf, sizeof(char), 30000, fi);
	fwrite(buf, sizeof(char), size, fo);

	entry[i].end = ftell(fo);

	fclose(fi);
      }
    }

    entry[i].type = FLAGS | file[i].type;
  }

  end_rom = ftell(fo);

  fseek(fo, bufpos, SEEK_SET);

  for(i=0; i<maxfil; i++)
    putbufent(&entry[i], fo);

  fclose(fo);

  fprintf(stderr, "End of ROM: %d,%d (%d)\n", end_rom / 256, end_rom % 256, end_rom);

  exit(0);
}
