/* LUnix-assembler Version 1.10

   Written by Daniel Dallmann (aka Poldi) on a weekend in June 1996 :-)
   This piece of software is freeware ! So DO NOT sell it !!
   You may copy and redistribute this file for free, but only with this header
   unchanged in it. (Or you risc eternity in hell)

   If you've noticed a bug or created an additional feature, let me know.
   My (internet) email-address is Daniel.Dallmann@studbox.uni-stuttgart.de

luna-extension-history:

 Jun 26 1996 *poldi* added ".header" directive

 Jun 28 1996 *poldi* inproved error detection (recursive label definitions)
                     writing of label-list included (experimental)

 Jul  7 1996 *poldi* new assembler directives:
                     .newpage - align to start of (next) page
                     .buf n   - insert n bytes

                     In expressions "*" dissolves to value of PC
                     
                     LUnix-mode:
                     .code      - enter code-area (default after .header)
                     .data      - enter data-area (inserts $0c <code-address>)
                     .endofcode - enter data-area for rest of file
                                   (relocation ends here, $02 inserted)

                     Feature: - ".buf"s at the very end will not be included
                                into the output file (that saves discspace)
                              - header directive allocates 2 bytes at once if
                                ZP-label has a "."-prefix.

                     removed bugs: comments after header-directive now ok.

 Jul 28 1996 *poldi* brackets in expressions. eg. "#(2+3)-(9-(6-2))" is 
                     possible now.

                     new assembler directive:
                     .global label[,label,...] - make label(s) be global, 
                       after compilation a list of all globals and their value
                       is saved in outfilename+".labels".

                     - Hex values now lower or uppercase.

 Aug 22 1996 *poldi* unknown labels are now reported only once.
                     removed ugly bug with addressingmode 12 (A).

 Aug 24 1996 *poldi* rewritten commandline parsing:
                       -q = quiet-mode commandline-switch
                       -o output-file
                       -l label-file

 Sep 4  1996 *poldi* rewritten parsing of expressions.
                     support of (linkable) object code.

 Oct 28 1996 *poldi* fixed bug with ".buf"s at start of code.

 Nov 1  1996 *poldi* added "~" - invert operator
                     added &   - AND in expressions
                     added |   - OR  in expressions
                     [...] can be used the same way as (...) in expressions

 Dec 15 1996 *poldi* added support of LUnix-objects
                     fixed bug with absolute addresses in object mode
 ...

*/

#include <stdio.h>

#undef debug

#define line_len  100   /* max line-length */
#define label_max 1000  /* max number of used labels */
#define label_len 8     /* just an average for calculating max labelspace */
#define BEF_NUM   151
#define LABEL_LEN 20    /* max length of labels */

static FILE *infile;
static FILE *binfile;
static int  pass;
static int  final_pass;
static int  unknown_flag;
static int  alter_flag;
static int  org_lock;
static int  lunix_mode;
static int  header[64];
static int  _line;
static char line[line_len+1];
static char msgbuf[LABEL_LEN+20];
static int  pc;
static int  pc_end;
static int  pc_begin;
static int  code_length;
static int  fmark,rmark,fmarks,rmarks;
static char label_tab[label_max*label_len];
static int  label_pos[label_max];
static int  label_val[label_max];
static int  label_stat[label_max];
static int  label_tab_pos;
static int  labels;
static int  errors,warnings;
static int  labresolved_flag;
static int  data_flag;
static int  buf_bytes;
static int  quiet_mode;
static char *file_input;
static unsigned char plain_buf[128];
static int  plain_buf_len;

#define no_label       (-1)   /* label number, that will never reached */

#define fl_resolved    0x0001 /* set, if label's value is known */
#define fl_used        0x0002 /* set, if label is used somewhere */
#define fl_global      0x0004 /* set, if label is defined as global */
#define fl_external    0x0008 /* set, if label is undefined */

/* special defines for creation of object-files */

static int  object_mode;

#define fl_variable    0x0010 /* set, if expression depends on org-addr. */
#define fl_extdep      0x0020 /* set, if expr depends on an external label */
#define fl_takelo      0x0040 /* set, if lo-byte of variable value is used */
#define fl_takehi      0x0080 /* set, if hi-byte of variable value is used */

#define fl_valdefs     (fl_variable|fl_extdep|fl_takelo|fl_takehi)

/* table of assembler commands and their opcodes */

static char *beflst[BEF_NUM]=
                 {  "cpx","cpx","cpx","cpy","cpy","cpy","bit","bit",
                    "bcc","bcs","beq","bne","bmi","bpl","bvc","bvs",
                    "jmp","jmp","jsr","asl","asl","asl","asl","asl",
                    "lsr","lsr","lsr","lsr","lsr","rol","rol","rol",
                    "rol","rol","ror","ror","ror","ror","ror","clc",
                    "cld","cli","clv","sec","sed","sei","nop","rts",
                    "rti","brk","lda","lda","lda","lda","lda","lda",
                    "lda","lda","ldx","ldx","ldx","ldx","ldx","ldy",
                    "ldy","ldy","ldy","ldy","sta","sta","sta","sta",
                    "sta","sta","sta","stx","stx","stx","sty","sty",
                    "sty","tax","tay","txa","tya","txs","tsx","pla",
                    "pha","plp","php","adc","adc","adc","adc","adc",
                    "adc","adc","adc","sbc","sbc","sbc","sbc","sbc",
                    "sbc","sbc","sbc","inc","inc","inc","inc","dec",
                    "dec","dec","dec","inx","dex","iny","dey","and",
                    "and","and","and","and","and","and","and","ora",
                    "ora","ora","ora","ora","ora","ora","ora","eor",
                    "eor","eor","eor","eor","eor","eor","eor","cmp",
                    "cmp","cmp","cmp","cmp","cmp","cmp","cmp"          };

static unsigned char  befopc[BEF_NUM]=
                {   224, 236, 228, 192, 204, 196,  44,  36,
                    144, 176, 240, 208,  48,  16,  80, 112,
                     76, 108,  32,  14,  30,   6,  22,  10,
                     78,  94,  70,  86,  74,  46,  62,  38,
                     54,  42, 110, 126, 102, 118, 106,  24,
                    216,  88, 184,  56, 248, 120, 234,  96,
                     64,   0, 169, 173, 189, 185, 165, 181,
                    161, 177, 162, 174, 190, 166, 182, 160,
                    172, 188, 164, 180, 141, 157, 153, 133,
                    149, 129, 145, 142, 134, 150, 140, 132,
                    148, 170, 168, 138, 152, 154, 186, 104,
                     72,  40,   8, 105, 109, 125, 121, 101,
                    117,  97, 113, 233, 237, 253, 249, 229,
                    245, 225, 241, 238, 254, 230, 246, 206,
                    222, 198, 214, 232, 202, 200, 136,  41,
                     45,  61,  57,  37,  53,  33,  49,   9,
                     13,  29,  25,   5,  21,   1,  17,  73,
                     77,  93,  89,  69,  85,  65,  81, 201,
                    205, 221, 217, 197, 213, 193, 209                  };

static unsigned char  befatp[BEF_NUM]=
                {     1,   2,   5,   1,   2,   5,   2,   5,
                     10,  10,  10,  10,  10,  10,  10,  10,
                      2,  11,   2,   2,   3,   5,   6,  12,
                      2,   3,   5,   6,  12,   2,   3,   5,
                      6,  12,   2,   3,   5,   6,  12,  13,
                     13,  13,  13,  13,  13,  13,  13,  13,
                     13,  13,   1,   2,   3,   4,   5,   6,
                      8,   9,   1,   2,   4,   5,   7,   1,
                      2,   3,   5,   6,   2,   3,   4,   5,
                      6,   8,   9,   2,   5,   7,   2,   5,
                      6,  13,  13,  13,  13,  13,  13,  13,
                     13,  13,  13,   1,   2,   3,   4,   5,
                      6,   8,   9,   1,   2,   3,   4,   5,
                      6,   8,   9,   2,   3,   5,   6,   2,
                      3,   5,   6,  13,  13,  13,  13,   1,
                      2,   3,   4,   5,   6,   8,   9,   1,
                      2,   3,   4,   5,   6,   8,   9,   1,
                      2,   3,   4,   5,   6,   8,   9,   1,
                      2,   3,   4,   5,   6,   8,   9                  };  

static unsigned char arglen[14]=
               {    4, 2, 3, 3, 3, 2, 2, 2, 2, 2, 0, 3, 1, 1            }; 

void Howto()
{
  printf("Usage:\n");
  printf("  luna [-loq] sourcefile\n");
  printf("    -l labelfile = file to store globals in\n");
  printf("    -o outputfile (default is \"c64.out\" or \"c64.o\")\n");
  printf("    -q = quiet mode\n");
  printf("    -O = create object instead of executable\n");
  exit(1);
}

void error(char *text)
{
  if (errors==0 && warnings==0 && quiet_mode) 
    printf("luna: In file \"%s\":\n",file_input);

  printf(" error: %s in line %i\n", text, _line);
  errors++;
}

void warning(char *text)
{
  if (warnings==0 && errors==0 && quiet_mode)
    printf("luna: In file \"%s\":\n",file_input);

  printf(" warning: %s in line %i\n", text, _line);
  warnings++;
}

#ifdef debug

void print_flags(int flags)
{
  printf("(");
  if (flags&fl_resolved) printf("res ");
  if (flags&fl_used)     printf("usd ");
  if (flags&fl_global)   printf("glb ");
  if (flags&fl_external) printf("ext ");
  if (flags&fl_variable) printf("var ");
  if (flags&fl_extdep)   printf("edp ");
  if (flags&fl_takelo)   printf("lo ");
  if (flags&fl_takehi)   printf("hi");
  printf(")\n");
}

#endif

int is_sep(int c)
{
  if (c>='0' && c<='9') return 0;
  if (c>='a' && c<='z') return 0;
  if (c>='A' && c<='Z') return 0;
  if (c=='_') return 0;
  if (c=='.') return 0;
  return 1;
}

int Readline()
{
  static int x;
  static int i;

  x=fgetc(infile);
  i=0;
  while (i<line_len && x!='\n') {
    if (x==EOF) {
      if (i==0) return (EOF);
      line[i]='\0';
      return (i); }
    line[i]=x;
    i=i+1; 
    x=fgetc(infile); }
  line[i]='\0';
  if (x=='\n') return (i);
  printf("error: Line to long\n");
  exit(1); 
}

int nextchar(int i)
{
  while ( line[i]==' ' || line[i]=='\t' ) i++;
  return (i);
}

int nextsep(int i)
{
  while (line[i]!=' ' && line[i]!='\t' && line[i]!='\0' && line[i]!=';' 
                      && line[i]!=',') i++;
  return (i);
}

int strwant(int *i, char *text)
{
  static int j;
  j=0;
  *i=nextchar(*i);
  while (text[j]!='\0') {
    if (text[j]!=line[*i]) return 0; 
    *i=nextchar(*i+1);
    j=j+1; }
  return 1;
}

int str_cmp(char *string1, char *string2)
{
  static int i;
  i=0;
  while (string1[i]!='\0') {
    if (string1[i]!=string2[i]) return 0;
    i=i+1; }
  if (string2[i]!='\0') return 0;
  return 1;
}

void setlabel(char *labname, int val, int flags)
{
  static int i;

  if (flags&(fl_extdep)) {
    error("external in label-definition");
    return; }

  /* search for label in database */
  i=0;
  while (i<labels) {
    if (str_cmp(&label_tab[label_pos[i]],labname)) {

      if (pass==1) {
        sprintf(msgbuf,"dupplicated label \"%s\"",labname);
        error(msgbuf);
        return; }

      if ((flags&fl_resolved)==0) { 
        /* val of label is not valid, because there have been unknown
           labels in expression, so clear bit */
        label_stat[i]&=~fl_resolved;
        if (final_pass) {
          sprintf(msgbuf,"label \"%s\" is unresolvable",labname);
         error(msgbuf); }
#       ifdef debug
        printf("label \"%s\" stays undefined\n",labname);
#       endif
	    return; }

      if (label_val[i]!=val) {
        label_val[i]=val;
        if (final_pass) {
          sprintf(msgbuf,"divergent labelvalue (%s)",labname);
          error(msgbuf); }
        if (label_stat[i]&fl_variable) labresolved_flag=1;
        alter_flag=1; }
      label_stat[i]=(label_stat[i]&~fl_valdefs)|(flags&fl_valdefs);
      if ((label_stat[i]&fl_resolved)==0) { 
        labresolved_flag=1;
        label_stat[i]=label_stat[i]|fl_resolved; }

      if (final_pass && (label_stat[i]&fl_used)==0 ) {
        sprintf(msgbuf,"unused label \"%s\"",&label_tab[label_pos[i]]);
        warning(msgbuf); }

#     ifdef debug
      printf("set \"%s\"=%i ",labname,val); 
      print_flags(label_stat[i]);
#     endif
      return; }

    i=i+1; }

  i=0;
  while (labname[i]!='\0') {

    if (label_tab_pos+i>=label_max*label_len) {
      error("label-space overflow");
      return; }

    label_tab[label_tab_pos+i]=labname[i];
    i=i+1;

    if (i==LABEL_LEN) {
      error("label too long");
      break; }
    }

  if (labels>=label_max) error("too many labels");
  label_tab[label_tab_pos+i]='\0';
  label_pos[labels]=label_tab_pos;
  label_val[labels]=val;
  if (!(flags&fl_resolved)) 
    label_stat[labels]=0;           /* value undefined, not global */
  else { 
    label_stat[labels]=(flags&(fl_valdefs|fl_external))|fl_resolved;  /* not global nor used   */
    labresolved_flag=1; }
  labels=labels+1;
  label_tab_pos=label_tab_pos+i+1;
  alter_flag=1;
# ifdef debug
  printf("create \"%s\"=%i ",labname,val);
  print_flags(label_stat[labels-1]);
# endif
}

int getascii(int i,int *par)
{
  if (line[i]=='\"') {                            /*"*/
    error("Illegal character constant");
    *par=0;
    return i+1; }
  if (line[i]=='\\') {
    i=i+1;
    if (line[i]=='n') {
      *par=13;
      return i+1; }
    if (line[i]=='0') {
      *par=0;
      return i+1; }
    if (line[i]=='\\') {
      *par='\\';
      return i+1; }
    if (line[i]=='\"') {                         /*"*/
      *par=34;
      return i+1; }
    error("unknown char");
    *par=0;
    return i+1; }
  *par=line[i];
  /* convert lower to upper case... */
  if (*par>='a' && *par<='z') *par=*par-'a'+'A';
  else if (*par>='A' && *par<='Z') *par=*par-'A'+'a';

  return i+1; 
}

void setglobal(char *str)
{
  static int i;

# ifdef debug
  printf("global: \"%s\"\n",str);
# endif

  i=0;
  while (i<labels) {
   if (str_cmp(str,&label_tab[label_pos[i]])) {
     label_stat[i]|=fl_global; /* set global-bit */
     return; }
   i=i+1; }
  unknown_flag=1;
# ifdef debug
  printf("unknown global\n");
# endif
  if (pass>1) {
    sprintf(msgbuf,"undefined global \"%s\"",str);
    error(msgbuf); }
  return;
}

int getlabel(char *str)
{
  static int i;
  i=0;
  while (i<labels) {
   if (str_cmp(str,&label_tab[label_pos[i]])) {
     label_stat[i]|=fl_used;

#    ifdef debug
     if ((label_stat[i]&fl_resolved)==0) {
       printf("unresolved label in expression\n"); }
#    endif

     return i; }
   i=i+1; }
  unknown_flag=1;
# ifdef debug
  printf("label %s unknown\n",str);
# endif
  if (pass>1) {
    setlabel(str,0,fl_external|fl_resolved);
    labresolved_flag=1;
    if (!object_mode) {
      sprintf(msgbuf,"undefined label \"%s\" (only reported once)",str);
      error(msgbuf); }
    else {
#     ifdef debug
      printf("added \"%s\" as external label\n",str);
#     endif
      return labels-1; } }

  return no_label;
}

int getval(int i, int *val, int *flags, int *lab)
{
  static int cnt;
  static int hlp;
  static int tmp;
  static char str[8];

  cnt=*val=*flags=0;
  *lab=no_label;

  if (line[i]=='\0') {
    error("expression expected");
    return i; }

  if (line[i]=='*') {
    /* "*" means value of PC */
    i=i+1;
    *val=pc;
    *flags=fl_variable|fl_resolved;
    return i; }

  /* check if its the "+" or "-" shortcut */
  if (line[i]=='+') {
    /* is a "+..." shortcut */
    while (line[i]=='+') {
      cnt=cnt+1;
      i=i+1; }
    sprintf(str,"_+%i",fmark+cnt-1);
    hlp=getlabel(str);
    if (hlp!=no_label) {
      *val=label_val[hlp];
      *flags=fl_variable|fl_resolved; }
    return i; }

  if (line[i]=='-') {
    /* is a "-..." shortcut */
    while (line[i]=='-') {
      cnt=cnt+1;
      i=i+1; }
    sprintf(str,"_-%i",rmark-cnt);
    hlp=getlabel(str);
    if (hlp!=no_label) {
      *val=label_val[hlp];
      *flags=fl_variable|fl_resolved; }
    return i; }
  
  if (line[i]=='#' || (line[i]>='0' && line[i]<='9')) {
    /* get decimal value */
    if (line[i]=='#') i=nextchar(i+1);
    while (!is_sep(line[i])) {
      if (line[i]>='0' && line[i]<='9') hlp=line[i]-'0';
      else { cnt=0; break; }
      *val=*val*10+hlp;
      cnt=cnt+1;
      i=i+1; }
    if (cnt==0) error("decimal expected");
    else *flags=fl_resolved;
    return i; }

  if (line[i]=='$') {
    /* get hex value */
    i=nextchar(i+1);
    while (!is_sep(line[i])) {
      if (line[i]>='0' && line[i]<='9') hlp=line[i]-'0';
      else if (line[i]>='a' && line[i]<='f') hlp=line[i]-'a'+10;
      else if (line[i]>='A' && line[i]<='F') hlp=line[i]-'A'+10;
      else { cnt=0; break; }
      *val=*val*16+hlp;
      cnt=cnt+1;
      i=i+1; }
    if (cnt==0) error("hex expected");
    else *flags=fl_resolved;
    return i; }

  if (line[i]=='\"') {
    /* get character value */
    i=nextchar(i+1);
    i=getascii(i,val);
    if (strwant(&i,"\"")) *flags=fl_resolved;
    else error("unterminated character constant");
    return i; }

  if (line[i]=='%') {
    /* get binary value */
    i=nextchar(i+1);
    while (!is_sep(line[i])) {
      if (line[i]=='0') hlp=0;
      else if (line[i]=='1') hlp=1;
      else { cnt=0; break; }
      *val=*val*2+hlp;
      cnt=cnt+1;
      i=i+1; }
    if (cnt==0) error("binary expected");
    else *flags=fl_resolved;
    return i; }

  /* nothing of the obove, so it must be a label */
  cnt=i;
  while (!is_sep(line[cnt])) cnt=cnt+1;
  tmp=line[cnt];
  line[cnt]='\0';
  /* now try to find a label that matches */
  hlp=getlabel(&line[i]);
  if (hlp!=no_label) {
    *val=label_val[hlp];
    *flags=label_stat[hlp];
    if (*flags&fl_external) {
      *val=0;
      *lab=hlp;
      *flags=(*flags&~fl_variable)|fl_extdep; }
    }

  line[cnt]=tmp;
  return cnt;
}

int getterm(int i, int *val, int *flags, int *lab)
{
  i=nextchar(i);

  if (line[i]=='<') {
    /* take lowbyte of term */
    i=getterm(i+1, val, flags, lab);
    if (object_mode && *flags&(fl_variable|fl_extdep)) {
      if (*flags&(fl_takelo|fl_takehi)) error("invalid expression");
      else *flags|=fl_takelo; }
    if (!object_mode) *val=*val&255;
    return i; }

  if (line[i]=='>') {
    /* take highbyte of term */
    i=getterm(i+1, val, flags, lab);
    if (object_mode && *flags&(fl_variable|fl_extdep)) {
      if (*flags&(fl_takelo|fl_takehi)) error("invalid expression");
      else *flags|=fl_takehi; }
    if (!object_mode) *val=(*val>>8)&255;
    return i; }

  if (line[i]=='~') {
    /* invert result (8bit only) */
    i=getterm(i+1, val, flags, lab);
    if (object_mode && *flags&(fl_variable|fl_extdep))
      error("invalid expression");
    if (*val>255) error("\"~\" is no 16bit operator");
    else *val=255-*val;
  return i; }

  if (line[i]=='(') {
    i=getexpr(i+1, val ,flags, lab);
    if (!strwant(&i,")")) error("\")\" expected");
    return i; } 

  if (line[i]=='[') {
    i=getexpr(i+1, val ,flags, lab);
    if (!strwant(&i,"]")) error("\"]\" expected");
    return i; }

  return getval(i, val, flags, lab);
}
  
int getexpr(int i, int *val, int *flags, int *lab)
{
  int tmp_val;
  int tmp_flags;
  int tmp_lab;

# ifdef debug
  printf("expr:\"%s\"\n",&line[i]);
# endif

  i=getterm(i, val, flags, lab);

  while (1) {

    i=nextchar(i);
    if (line[i]=='+') {
      i=getterm(i+1, &tmp_val, &tmp_flags, &tmp_lab);
      if (!(tmp_flags&*flags&fl_resolved)) {
        *flags=0; continue; }
      if (!object_mode) *val+=tmp_val;
      else {
        if ((tmp_flags|*flags)&(fl_takelo|fl_takehi)) 
          error("invalid expression");
        else if (!(*flags&fl_valdefs)) {
          *val+=tmp_val;
          *flags=tmp_flags;
          *lab=tmp_lab; }
        else if (!(tmp_flags&fl_valdefs)) {
          *val+=tmp_val; }
        else {
          *flags=0;
          error("invalid expression"); }
	    }
      continue; }

    if (line[i]=='-') {
      i=getterm(i+1, &tmp_val, &tmp_flags, &tmp_lab);
      if (!(tmp_flags&*flags&fl_resolved)) {
        *flags=0; continue; }
      if (!object_mode) *val=*val-tmp_val;
      else {
        if ((tmp_flags|*flags)&(fl_takelo|fl_takehi)) 
          error("invalid expression");
        else if (!(*flags&fl_valdefs)) {
          *val-=tmp_val;
          *flags=tmp_flags;
          *lab=tmp_lab; }
        else if (!(tmp_flags&fl_valdefs)) {
          *val-=tmp_val; }
        else if (*flags&tmp_flags&fl_variable) {
          *val-=tmp_val;
          *flags=fl_resolved; }
        else {
          *flags=0;
          error("invalid expression"); }
	    }
      continue; }

    if (line[i]=='|') {
      i=getterm(i+1, &tmp_val, &tmp_flags, &tmp_lab);
      if (!(tmp_flags&*flags&fl_resolved)) {
        *flags=0; continue; }
      if (!object_mode) *val=*val|tmp_val;
      else {
        if ((tmp_flags|*flags)&(fl_takelo|fl_takehi|fl_extdep|fl_variable)) 
          error("invalid expression");
        else *val=*val|tmp_val; }
      continue; }

    if (line[i]=='&') {
      i=getterm(i+1, &tmp_val, &tmp_flags, &tmp_lab);
      if (!(tmp_flags&*flags&fl_resolved)) {
        *flags=0; continue; }
      if (!object_mode) *val=*val&tmp_val;
      else {
        if ((tmp_flags|*flags)&(fl_takelo|fl_takehi|fl_extdep|fl_variable)) 
          error("invalid expression");
        else *val=*val&tmp_val; }
      continue; }

    break; }

# ifdef debug
  printf("expression is: val=%i, lab=%i, flags=",*val,*lab);
  print_flags(*flags);
# endif

  return i;
}


int getasspar(int i, int *mode, int *par, int *flags, int *lab)
{
  static int tmp;
  i=nextchar(i); 
  *flags=0;
  if (line[i]=='\0') { *mode=13; return i; }
  if (line[i]==',' || line[i]==';') { *mode=13; return i; }

  if (line[i]=='(') {
    /* (expr,x) or (expr),y or (expr) */
    i=nextchar(i+1);
    if (line[i]=='\0') { *mode=13; error("stray \"(\" found"); return i; }
    i=getexpr(i,par,flags,lab);
    i=nextchar(i);
    if (line[i]=='\0' || line[i]==';') { 
      *mode=2; error("missing \")\""); return i; }

    if (line[i]==')') {
      i=nextchar(i+1);
      if (line[i]==',') {
        /* must be (expr),y */
        *mode=9;
        if (!strwant(&i,",y")) error("expected \",y\"");
        return i; }
      else {
        /* must be (expr) */
        *mode=11;
        return i; } }

    /* must be (expr,x) */
    *mode=8;
    if (!strwant(&i,",x)")) error("expected \",x)\"");
    return i; }

  if (line[i]=='a' && (line[i+1]==' '|| line[i+1]=='\t' || line[i+1]=='\0'
                                     || line[i+1]==';'  || line[i+1]==':' )) {
    *mode=12; return nextchar(i+1); }

  if (line[i]=='#') {
    *mode=1;
    i=getexpr(i+1,par,flags,lab);
    i=nextchar(i);
    return i; }

  if (tmp=(line[i]=='.')) i=nextchar(i+1);
  i=getexpr(i,par,flags,lab);
  i=nextchar(i);
  if (   !((*par&0xff00)==0 && *flags&fl_resolved) 
      || tmp
      || object_mode&&(*flags&fl_variable)
      || *flags&fl_extdep) 
       *mode=2; /* word */
  else *mode=5; /* byte */

  if (line[i]!='\0') {
    if (line[i]==',') {
      i=nextchar(i+1);
      if (line[i]=='\0') { error("stray \",\""); return i; }
      if (line[i]=='x') *mode=*mode+1;
      else if(line[i]=='y') *mode=*mode+2;
      else error("unknown index");
      i=nextchar(i+1); }}
  return i;
}

void raw_put(int x)
{
# ifdef debug
  printf("*** writing %i\n",x);
# endif

  if ((x&0xff00)!=0) { 
    error("expression out of range");
    return; }
  if (fputc(x,binfile)==EOF) {
    printf("i/o-error while writing to outputfile\n");
    exit(1); }
  code_length++;
}

void cleanup_plain_buf()
{
  static int i;

  if (plain_buf_len==0) return;
  raw_put(plain_buf_len);
  i=0;
  while (i<plain_buf_len) raw_put((int) plain_buf[i++]);
  plain_buf_len=0;
}
  
void putbyte(int i,int flags, int lab)
{
  static int x,y;

  if (final_pass) {
    if (org_lock==0) {
      pc_begin=pc;
      if (object_mode) {

        if (lunix_mode) raw_put('O'); else raw_put('o');
        raw_put('b');
        raw_put('j');

#       ifdef debug

        printf("object header\n");
        printf("  globals:\n");

#       endif

        x=0;
        while (x<labels) {
          if (label_stat[x]&fl_global) {
#           ifdef debug
            printf("    %s=%i\n",&label_tab[label_pos[x]],label_val[x]);
#           endif
            fprintf(binfile,"%s",&label_tab[label_pos[x]]);
            code_length+=strlen(&label_tab[label_pos[x]]);
            raw_put(0);
            raw_put(label_val[x]&0xff);
            raw_put((label_val[x]>>8)&0xff); }
          x++; }
#       ifdef debug
        printf("code-length=%i\n",pc_end);
        printf("  externals=\n");
#       endif
        raw_put(0);
        raw_put(pc_end&0xff);
        raw_put((pc_end>>8)&0xff);
        x=0; y=0;
        while (x<labels) {
          if (label_stat[x]&fl_external) {
#           ifdef debug
            printf("    %2i %s\n",y,&label_tab[label_pos[x]]);
#           endif
            fprintf(binfile,"%s",&label_tab[label_pos[x]]);
            code_length+=strlen(&label_tab[label_pos[x]]);
            raw_put(0);
            label_val[x]=y++; }
          x++; }
#       ifdef debug
        printf("\n");
#       endif
        raw_put(0); }
      else {
        if (!lunix_mode) {
          x=pc-buf_bytes;
#         ifdef debug
          printf("code starts at %i\n",x);
#         endif
          if (x==0) warning("missing org directive, assume \"org $0000\"");
          raw_put(x&0xff);
          raw_put((x>>8)&0xff); }
        else {
          raw_put(0xff);
          raw_put(0xff); } }
	}
#   ifdef debug
    printf("### byte=%i lab=%i flags=",i,lab);
    print_flags(flags);
#   endif

    if (!object_mode) {
      /* insert buffer bytes if there are */
      if (buf_bytes!=0) {
#       ifdef debug
        printf("### inserting %i-buffer bytes\n",buf_bytes);
#       endif
        x=0;
        while (x<buf_bytes) {
          raw_put(0);
          x=x+1; } }
      /* continue with actual stuff */
      raw_put(i); }
    else {
      /* insert buffer bytes if there are */
      if (buf_bytes!=0) {
#       ifdef debug
        printf("### inserting %i-buffer bytes\n",buf_bytes);
#       endif
        x=0;
        while (x<buf_bytes) {
          plain_buf[plain_buf_len++]=0;
          if (plain_buf_len==127) cleanup_plain_buf();
          x=x+1; } }
      /* continue with actual stuff */
      if ((flags&fl_valdefs)==0) {
        plain_buf[plain_buf_len++]=(unsigned char) i;
        if (plain_buf_len==127) cleanup_plain_buf(); }
      else {
        cleanup_plain_buf();
        if (flags&fl_takelo) x=0x01;
        else if (flags&fl_takehi) x=0x02;
        else x=0x03;
        if (flags&fl_variable) {
          raw_put(0x80|x);
          raw_put(i&0xff);
          raw_put((i>>8)&0xff); }
        else if (flags&fl_external) {
          if (i==0) {
            raw_put(0xc0|x);
            raw_put(label_val[lab]&0xff);
            raw_put((label_val[lab]>>8)&0xff); }
          else {
            raw_put(0xd0|x);
            raw_put(label_val[lab]&0xff);
            raw_put((label_val[lab]>>8)&0xff);
            raw_put(i&0xff);
            raw_put((i>>8)&0xff); } }
        else error("internal assembler error"); }
      }
    }
  org_lock=1;
  pc++;
  buf_bytes=0;
}

void putword(int word,int flags, int lab)
{
  if (!object_mode || (flags&fl_valdefs)==0) {
    putbyte(word&0xff,fl_resolved,0);
    putbyte((word>>8)&0xff,fl_resolved,0);
    return; }

  if (flags&(fl_takelo|fl_takehi)) {
    error("byte instead of word");
    return; }

  putbyte(word,flags,lab);
  pc++;
}

void writebef(int opcode, int befatp, int par, int flags, int lab)
{
  static int tmp;

  tmp=arglen[befatp];

  if (tmp==1) putbyte(opcode,fl_resolved,0);
  else if (tmp==2) {
    putbyte(opcode,fl_resolved,0);
    putbyte(par,flags,lab); }
  else if (tmp==3) {
    putbyte(opcode,fl_resolved,0);
    putword(par,flags,lab); } 
  else if (tmp==0) {
    /* relative jump, if out of range then replace by long
       range jump.

         eg. beq label -> bne *+5
                          jmp label   */

    if ((flags&fl_resolved)==0) par=pc+2; 
                    /* don't make far jumps, if address in unknown */
    if (flags&fl_extdep) {
      error("conditional jump to external");
      return; }
    if (par>=pc+2) {
      tmp=par-pc-2;
      if (tmp>127) tmp=256;
      else {
        putbyte(opcode,fl_resolved,0);
        putbyte(tmp,fl_resolved,0); } }
    else {
      tmp=pc+2-par;
      if (tmp>128) tmp=256;
      else {
        putbyte(opcode,fl_resolved,0);
        putbyte(256-tmp,fl_resolved,0); } }
    if (tmp==256) {
      if (pass>1) warning("relative jump out of range");
      putbyte(opcode^32,fl_resolved,0);
      putbyte(3,fl_resolved,0);
      putbyte(76,fl_resolved,0);
      putword(par,flags,lab); }
    }
  else printf("internal error tmp=%i\n",tmp);
}

void setsigjmp(char *signame, int pos)
{
  static int i;
 
  i=0;
  while (i<labels) {
    if (str_cmp(signame,&label_tab[label_pos[i]])) break;
    i=i+1; }
  if (i==labels) {
    if (final_pass) {
      if (!quiet_mode) 
        printf(" note: ignored unused system vector \"%s\"\n",signame); }
    return; }
  header[pos*3+8]=76;
  header[pos*3+9]=label_val[i]&255;
  if ((label_stat[i]&fl_resolved)==0) {
    sprintf(msgbuf,"undefined systemvector \"%s\"",signame);
    error(msgbuf); }
  label_stat[i]=label_stat[i]|fl_used;
  header[pos*3+10]=(label_val[i]>>8)&255;
} 

void main(int argc, char **argv)
{
  static int llen;
  static int i,j,p,q;
  static char str[100];
  static int mode;
  static int par;
  static int tmp;
  static int dseg_count;
  extern char *file_input;
  static char *file_output;
  static char *file_labels;
  static int  flags,lab;

  pass=0;
  final_pass=0;
  alter_flag=1;
  unknown_flag=1;
  quiet_mode=0;
  labels=0;
  rmarks=fmarks=0;
  errors=warnings=0;
  label_tab_pos=0;
  labresolved_flag=1;
  file_input=file_output=file_labels=NULL;
  object_mode=0;
  plain_buf_len=0;
  code_length=0;

  i=0;
  while (i<64) {
    header[i]=0;
    i=i+1; }

  i=1;
  while ( i<argc ) {
    if (argv[i][0]=='-') {
      j=1;
      while (argv[i][j]!='\0') {
        switch (argv[i][j]) {
          case 'q': { quiet_mode=1; break; }
          case 'l': { i++; file_labels=argv[i]; j=0; break; }
          case 'o': { i++; file_output=argv[i]; j=0; break; }
          case 'O': { object_mode=1; break; }
          default:  Howto();
          }
        if (i==argc) Howto();
        if (j==0) break;
        j++; }
	} else {
      if (file_input!=NULL) Howto();
      file_input=argv[i]; }
    i++; }

  if (file_input==NULL) { printf("%s: No input file\n",argv[0]); exit(1); }
  if (file_output==NULL) 
    if (object_mode) file_output="c64.o"; else file_output="c64.out";

  if (!quiet_mode) printf("Luna 6502/10-cross-assembler version 1.10\n");

  if (str_cmp(file_input,file_output)) {
    printf("warning: sourcefile=destfile ??\n");
    exit(1); }
   
  do {
    if ((alter_flag==0 || unknown_flag==0) && !labresolved_flag) final_pass=1;
#   ifdef debug
    printf("runtimeflags: alter:%i unknown:%i labresolved:%i\n",alter_flag,unknown_flag,labresolved_flag);
#   endif
    alter_flag=unknown_flag=0;
    _line=0;
    rmark=fmark=0;
    pc=0;
    org_lock=0;
    labresolved_flag=0;
    data_flag=0;
    dseg_count=0;
    buf_bytes=0;
    pass=pass+1;

    infile=fopen(file_input,"r");
    if (infile==NULL) {
      printf("%s: error: Can't open \"%s\"\n",argv[0],file_input);
      exit(1); }

#   ifdef debug
    printf("\n\n");
#   endif
    if (!final_pass && !quiet_mode) printf("Pass %i\n",pass);
    else { 
      if (!quiet_mode) printf("Final pass\n");
      binfile=fopen(file_output,"w");
      if (binfile==NULL) {
        printf("%s: can't create \"%s\"\n",argv[0],file_output); 
        exit(1); }
      } 
    while ( (llen=Readline())!=EOF ) {
      _line=_line+1;
#     ifdef debug
      printf("line \"%s\"\n",line);
#     endif
      if (llen=0) continue;
      if (line[0]==';') continue;
      
      /* extract first identifier */

      i=0;
      j=0;
      while (line[i]!='\0') {
        i=nextchar(j);
        if (line[i]=='\0') continue;
        if (line[i]==':') { 
          j=nextchar(i+1);
          continue; }                
        j=nextsep(i);
#       ifdef debug
        printf("working at \"%s\"\n",&line[i]);
#       endif
        if (line[i]==';') break;  /* keine Befehle mehr in dieser Zeile */
        if (line[i]=='.') {

          /* special assembler commands like .byte .word .asc .head */

          i=i+1;
          if (line[i]=='b' && line[i+1]=='y') {

            /* assume .byte */

            while (1) {
              j=nextchar(j);
              j=getexpr(j,&par,&flags,&lab);
              if (par>255) error("byte out of range");
              putbyte(par,flags,lab);
              if (line[j]!=',') break;
              if (lunix_mode && data_flag==0) error("data in code-area");
              j=j+1; }
		    continue; }

          if (line[i]=='w') {

            /* assume word */

            if (lunix_mode && data_flag==0) error("data in code-area");
            while (1) {
              j=nextchar(j);
              j=getexpr(j,&par,&flags,&lab);
              putword(par,flags,lab);
              if (line[j]!=',') break;
              j=j+1; }
            continue; }

          if (line[i]=='a') {

            /* assume asc */

            if (!strwant(&j,"\"")) error("\" expected");
            if (lunix_mode && data_flag==0) error("string in code-area");
            while (1) {
              if (line[j]=='\"') {                           /*"*/
                j=j+1; 
                break; }
              if (line[j]=='\0') {
                error("unterminated string");
                break; }
              j=getascii(j,&par);
              putbyte(par,fl_resolved,0); }
            continue; }

          if (line[i]=='h') {

            /* LUnix header creator */

            if (org_lock) {
              error("nested org (in header)");
              continue; }
            if (buf_bytes!=0) error(".header after .buf");
            if (!object_mode) pc=4096; /* defaul org-address */
            pc_begin=pc;
            if (pass==1) pc_end=pc_begin;
            lunix_mode=1;
            j=nextchar(j);
            if (!strwant(&j,"\"")) {
              error("CMD-name expected");
              continue; }
            p=0;
            while (1) {
              if (line[j+p]=='\"') {                   /*"*/
                header[56+p]=0;
                p=p+1;
                break; }
              if (is_sep(line[j+p])) {
                error("unterminated CMD-name");
                break; }
              header[56+p]=line[j+p];
              p=p+1;
              if (p==8) {
                error("CMD-name too long (8 chars max)");
                break; }
            }
            j=j+p;
            p=217; /* default zeropage startaddress */
            while (1) {
              j=nextchar(j);
              if (line[j]=='\0') break;
              if (line[j]==';')  break;
              if (line[j]!=',') {
                error("syntax error");
                break; }
              j=nextchar(j+1);
              q=nextsep(j);
              tmp=line[q];
              line[q]='\0';
              if (line[j]=='.') {       /* if ZP-label has a "." prefix,then */
                setlabel(&line[j+1],p,fl_resolved);
                                        /* 2 bytes are allocated instead of  */
                p=p+2; }                /* just one                          */
              else {
                setlabel(&line[j],p,fl_resolved);
                p=p+1; }
              line[q]=tmp;
              j=q; }
            header[1]=1;
            header[4]=200-(p-217);
            header[5]=217;
            header[6]=p-217;
            header[2]=1+((pc_end-pc_begin)>>8);
            header[8]=28;
            header[9]=64;
            header[10]=0;
            /* insert optional sig-jumps */
            setsigjmp("_sig.userbreak",6);
            setsigjmp("_sig.killedparent",7);
            setsigjmp("_sig.killedchild",8);
            setsigjmp("_cleanup",9);
            i=0;
            while (i<64) {
              putbyte(header[i],fl_resolved,0);
              i=i+1; }
            i=getlabel("_init");
            flags=0; lab=0;
            if (i!=no_label) {
              par=label_val[i];
              flags=label_stat[i];
              if (flags&fl_external) error("missing _init label"); }
            else par=0;
            if (i!=no_label) i=label_val[i]; else i=0;
            putbyte(169,fl_resolved,0);
            putbyte((pc_begin)>>8 &255,fl_resolved,0);
            putbyte(32,fl_resolved,0);        /* instert lda #>pc_begin   */
            putbyte(81,fl_resolved,0);        /*         jsr $9051        */
            putbyte(144,fl_resolved,0);       /*         jmp _init        */
            putbyte(76,fl_resolved,0);
            putword(i,flags,lab);
            setlabel("_base",pc_begin,fl_resolved|fl_variable);
            continue; }

          if (line[i]=='b' && line[i+1]=='u') {
 
            /* assume .buf */
         
            if (lunix_mode && data_flag==0) warning(".buf in code-area");
            j=nextchar(j);
            j=getexpr(j,&par,&flags,&lab);
            buf_bytes=buf_bytes+par; /* delayed insertion, because we don't
                                        need buf-bytes at the very end. */
            pc=pc+par;
            continue; }

          if (line[i]=='d') {
 
            /* assume .data */
_do_data:
            if (data_flag!=0) continue;
            data_flag=1;   
            dseg_count=dseg_count+1;
            sprintf(str,"__d%i",dseg_count);
            i=getlabel(str);
            flags=0; lab=0;
            if (i!=no_label) {
              par=label_val[i];
              flags=label_stat[i];
              if (flags&fl_external) error(".data without .code"); }
            else par=0;
            writebef( 12, 2, par, flags, lab);
            continue; }
 
          if (line[i]=='c') {
            
            /* assume .code */

            if (data_flag==0) continue;
            if (data_flag==2) error("code after .endofcode");
            data_flag=0;
            sprintf(str,"__d%i",dseg_count);
            setlabel( str,pc,fl_resolved|fl_variable);
            continue; }

          if (line[i]=='e') {

            /* assume .endofcode */
 
            if (object_mode) goto _do_data;
              /* library calls will be appended to this file, so
                 .endofcode would be a big mistake */
            if (data_flag==1) {
              sprintf(str,"__d%i",dseg_count);
              setlabel( str,pc,fl_resolved|fl_variable); }
            putbyte(2,fl_resolved,0);
            data_flag=2; 
            continue; }

          if (line[i]=='n') {
         
            /* assume .newpage */
           
            if (lunix_mode && data_flag==0) error(".newpage in code-area");
            if (object_mode) warning(".newpage might not work in objectmode");
            i=(pc & 255);
            if (i!=0) i=256-i;
            if (final_pass && !quiet_mode) 
              printf(" note: %i unused bytes because of .newpage\n",i);
            buf_bytes=buf_bytes+i;
            pc=pc+i;
            continue; }

          if (line[i]=='g') {

            /* assume .global */

            j=nextchar(j);
            while (1) {
              q=nextsep(j);
              tmp=line[q];
              line[q]='\0';
              setglobal(&line[j]);
              line[q]=tmp;
              j=nextchar(q);
              if (line[j]=='\0') break;
              if (line[j]==';')  break;
              if (line[j]!=',') {
                error("syntax error");
                break; }
              j=nextchar(j+1); }
            continue; }
         
		}
        if (line[j-1]==':') {
          /* label: */
         line[j-1]='\0';
         setlabel(&line[i],pc,fl_resolved|fl_variable);
         continue; }
        if (j-i==1) {
          /* one char prefix */
          if (line[i]=='+') {
            sprintf(str,"_+%i",fmark);
            fmark=fmark+1;
            setlabel(str,pc,fl_resolved|fl_variable); }
          else if (line[i]=='-') {
            sprintf(str,"_-%i",rmark);
            rmark=rmark+1;
            setlabel(str,pc,fl_resolved|fl_variable); }
          else error("syntax error");
          continue; }
        if (j-i==3) {  
          p=0;
          while (p<BEF_NUM) {
            if (beflst[p][0]==line[i]   &&
                beflst[p][1]==line[i+1] &&
                beflst[p][2]==line[i+2]   ) break;
            p=p+1; }
          if (p!=BEF_NUM) {
            /* found assembler command */
            j=getasspar(j,&mode,&par,&flags,&lab); 
            /* find perfect match */
            q=p;
            tmp=BEF_NUM;
            while (beflst[q][0]==line[i]   &&
                   beflst[q][1]==line[i+1] &&
                   beflst[q][2]==line[i+2]   ) {
              if (mode==befatp[q]) tmp=q;
              if (mode>=5 && mode<=7 && mode-3==befatp[q]) tmp=q;
              if (befatp[q]==10) if (mode==2 || mode==5) tmp=q; 
              q=q+1;
              if (q==BEF_NUM) break; }    
            if (tmp==BEF_NUM) {
              error("addressing mode not supported");
              continue; }
            else {
              writebef(befopc[tmp],befatp[tmp],par,flags,lab);
              continue; }
          }
          /* Befehl besteht aus 3 Buchstaben wurde, aber nicht gefunden */
          if (line[i]=='o' && line[i+1]=='r' && line[i+2]=='g' ) {
            /* org-command */
            j=nextchar(j);
            j=getexpr(j,&par,&flags,&lab);
            if (!object_mode) pc=par; else pc=0;
            if (org_lock) error("nested org");
            continue; }
        }
      /* kein solcher Befehl gefunden */ 
      tmp=nextchar(j);
      line[j]='\0';
      if (line[tmp]=='e' && line[tmp+1]=='q' && line[tmp+2]=='u') {
        /* label equ expression */
        j=nextchar(nextsep(tmp));
        j=getexpr(j,&par,&flags,&lab);
        setlabel(&line[i],par,flags);
        continue; }
      error("unknown command");        
      }
    }

  /* pass done */

  fclose(infile);
   if (lunix_mode && data_flag!=2) {
     if (!object_mode) error("missing .endofcode directive");
     if (data_flag==1) {
       /* add .code to end of file */
       data_flag=0;
       sprintf(str,"__d%i",dseg_count);
       setlabel( str,pc,fl_resolved|fl_variable); }
     }
  if (!object_mode) i=pc; else i=pc+buf_bytes;
  if ((pc_end!=i || labresolved_flag) && final_pass) {
    pc_end=i;
    if (!quiet_mode) printf("  sorry, need another pass\n");
    fclose(binfile);
    final_pass=0; }
  pc_end=i;

  if (errors!=0) {
    printf("summary: %i error",errors);
    if (errors>1) printf("s");
    printf(", stopped after pass %i\n",pass);
    exit(1); }
  }
  while (!final_pass);

  if (buf_bytes!=0) {
    if (object_mode) {
      i=0;
      while (i<buf_bytes) {
        plain_buf[plain_buf_len++]=0;
        if (plain_buf_len==127) cleanup_plain_buf();
        i=i+1; }
      }
    else if (!quiet_mode)
      printf(" note: last %i buffer-bytes not saved in image\n",buf_bytes);
    }

  if (object_mode) {
    cleanup_plain_buf();
    raw_put(0); }
  
  fclose(binfile);

  /* all done */                    
  if (!quiet_mode) {
    printf("done, %i labels, %i bytes labelspace, %i bytes of code",labels,label_tab_pos,pc_end-pc_begin);
    if (lunix_mode) printf(" (LUnix)");
    if (object_mode) printf(" (objectcode %i bytes)",code_length);
    printf("\n"); }

  /* optional: writing of labellist */

  if (file_labels!=NULL) {
    binfile=fopen(file_labels,"w");
    i=0;
    while (i<labels) {
      if ( (label_stat[i]&fl_global)!=0 ) 
        fprintf(binfile,"%s \tequ %i\n",&label_tab[label_pos[i]],label_val[i]);
      i=i+1; }
    fclose(binfile); }

  exit(0);
}
